class PageObserver {
    constructor(options = {}) {
        this.threshold = options.threshold || 0.2;
        this.rootMargin = options.rootMargin || '0px';
        this.observer = null;
        this.observations = new Map(); // Track observation data
        this.initObserver();
    }

    initObserver() {
        this.observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const element = entry.target;
                    const observationData = this.getObservationData(element);
                    
                    // Update timing data
                    observationData.triggers++;
                    observationData.lastTrigger = Date.now();
                    observationData.timeOnPage = observationData.lastTrigger - observationData.firstTrigger;

                    // Call callback with observation data
                    if (element.dataset.observerCallback) {
                        window[element.dataset.observerCallback](element, observationData);
                    }
                }
            });
        }, {
            threshold: this.threshold,
            rootMargin: this.rootMargin
        });
    }

    getObservationData(element) {
        if (!this.observations.has(element)) {
            this.observations.set(element, {
                firstTrigger: Date.now(),
                lastTrigger: Date.now(),
                timeOnPage: 0,
                triggers: 0
            });
        }
        return this.observations.get(element);
    }

    observe(selector, callback) {
        const elements = document.querySelectorAll(selector);
        if (elements.length === 0) return;

        elements.forEach(element => {
            if (callback) {
                const callbackName = `scrollCallback_${Math.random().toString(36).substr(2, 9)}`;
                window[callbackName] = callback;
                element.dataset.observerCallback = callbackName;
            }
            this.observer.observe(element);
        });
    }

    unobserve(selector) {
        const elements = document.querySelectorAll(selector);
        elements.forEach(element => {
            this.observer.unobserve(element);
            this.observations.delete(element);
            if (element.dataset.observerCallback) {
                delete window[element.dataset.observerCallback];
                delete element.dataset.observerCallback;
            }
        });
    }
}
